
//DIB Section support for WinGlide
//Copr. 1998, Chris Dohnal (cwdohnal@ucdavis.edu)

#include <windows.h>
#include "msgbox.h"

BOOL CreateDIB(HWND, DWORD, DWORD);
VOID FreeDIB(VOID);
LPVOID GetDIBBuffer(VOID);
LONG GetDIBWidth(VOID);
LONG GetDIBHeight(VOID);
LONG GetDIBPitch(VOID);
VOID DrawDIB(VOID);

VOID CreateDIBSectionPalette(VOID);

LPVOID pDIBData = NULL;
volatile LONG lDIBWidth, lDIBHeight;
LONG lDIBPitch;
HWND hDIBWnd = NULL;
HBITMAP hDIB = NULL;
HBITMAP hOldBitmap;
HDC hDIBDC = NULL;
HPALETTE hDIBPalette = NULL;

BOOL CreateDIB(HWND hWnd, DWORD dwWidth, DWORD dwHeight) {
	LONG lAlignedDIBWidth;
	struct {
		BITMAPINFOHEADER bmiHeader; 
		DWORD bmiColors[3];
	} bmi;

	//Free the previous DIB section if it exists
	if (hDIB != NULL) {
		FreeDIB();
	}
	
	//Save information about the DIB
	hDIBWnd = hWnd;
	lDIBWidth = (LONG)dwWidth;
	lDIBHeight = (LONG)dwHeight;

	//Alignment
	lAlignedDIBWidth = ((lDIBWidth + 3) & ~3);

	//Save the DIB pitch in bytes
	lDIBPitch = lAlignedDIBWidth << 1;
	
	bmi.bmiHeader.biSize = sizeof(BITMAPINFOHEADER);
	bmi.bmiHeader.biWidth = lAlignedDIBWidth;
	bmi.bmiHeader.biHeight = -lDIBHeight;
	bmi.bmiHeader.biPlanes = 1;
	bmi.bmiHeader.biBitCount = 16;
	bmi.bmiHeader.biCompression = BI_BITFIELDS;
	bmi.bmiHeader.biSizeImage = (lAlignedDIBWidth * lDIBHeight) << 1;
	bmi.bmiHeader.biXPelsPerMeter = 0;
	bmi.bmiHeader.biYPelsPerMeter = 0;
	bmi.bmiHeader.biClrUsed = 0;
	bmi.bmiHeader.biClrImportant = 0;
	bmi.bmiColors[0] = 0x0000F800;
	bmi.bmiColors[1] = 0x000007E0;
	bmi.bmiColors[2] = 0x0000001F;

	//Create a device context that is compatible with the current screen
	hDIBDC = CreateCompatibleDC(NULL);

	//Create the DIB Section
	hDIB = CreateDIBSection(hDIBDC, (BITMAPINFO *)&bmi, DIB_RGB_COLORS, &pDIBData, NULL, 0);
	if (hDIB == NULL) {
		return FALSE;
	}

	//Select the DIB into the device context
	hOldBitmap = SelectObject(hDIBDC, hDIB);

	//Setup the palette
	CreateDIBSectionPalette();

	return TRUE;
}

VOID FreeDIB(VOID) {
	//Free the device context and the DIB section
	if (hDIB != NULL) {
		SelectObject(hDIBDC, hOldBitmap);
		DeleteObject(hDIB);
		hDIB = NULL;
		DeleteDC(hDIBDC);
		hDIBDC = NULL;
	}
	//Free the palette
	if (hDIBPalette != NULL) {
		DeleteObject(hDIBPalette);
		hDIBPalette = NULL;
	}
	return;
}

LPVOID GetDIBBuffer(VOID) {
	return pDIBData;
}

LONG GetDIBWidth(VOID) {
	return lDIBWidth;
}

LONG GetDIBHeight(VOID) {
	return lDIBHeight;
}

LONG GetDIBPitch(VOID) {
	//Return the pitch in bytes
	return lDIBPitch;
}

VOID DrawDIB(VOID) {
	HDC hDC;

	hDC = GetDC(hDIBWnd);

	SelectPalette(hDC, hDIBPalette, FALSE);
	RealizePalette(hDC);

	BitBlt(hDC, 0, 0, lDIBWidth, lDIBHeight, hDIBDC, 0, 0, SRCCOPY);

	ReleaseDC(hDIBWnd, hDC);

	return;
}

VOID DrawStretchedDIB(VOID) {
	HDC hDC;
	RECT rect;
	LONG lWidth, lHeight;

	//Calculate the width and height of the destination rectangle
	GetClientRect(hDIBWnd, &rect);
	lWidth = rect.right - rect.left;
	lHeight = rect.bottom - rect.top;
		
	hDC = GetDC(hDIBWnd);

	SetStretchBltMode(hDC, COLORONCOLOR);

	SelectPalette(hDC, hDIBPalette, FALSE);
	RealizePalette(hDC);

	StretchBlt(hDC, 0, 0, lWidth, lHeight, hDIBDC, 0, 0, lDIBWidth, lDIBHeight, SRCCOPY);

	ReleaseDC(hDIBWnd, hDC);

	return;
}

VOID CreateDIBSectionPalette(VOID) {
	HDC hDC;
	DWORD dw;
	struct {
		WORD palVersion; 
		WORD palNumEntries; 
		PALETTEENTRY palPalEntry[256];
	} palette;

	//Create a grayscale palette
	palette.palVersion = 0x300;
	palette.palNumEntries = 256;

	for (dw = 0; dw < 256; dw++) {
		palette.palPalEntry[dw].peRed = (BYTE)dw;
		palette.palPalEntry[dw].peGreen = (BYTE)dw;
		palette.palPalEntry[dw].peBlue = (BYTE)dw;
		palette.palPalEntry[dw].peFlags = 0;
	}
	
	hDIBPalette = CreatePalette((LOGPALETTE *)&palette);
	hDC = GetDC(NULL);
	SelectPalette(hDC, hDIBPalette, FALSE);
	RealizePalette(hDC);
	UpdateColors(hDC);
	ReleaseDC(NULL, hDC);

	return;
}
